<?php
class ScManager {

    static function getScIdList() {
        $sc_id_list = pg_fetch_all_columns(
                            DbManager::query( 'SELECT sc_id FROM tb_sc_info' ),
                            0 ) ;
        return array_slice( $sc_id_list, 1 ) ;
    }

    static function getPowerState( $sc_id ) {
        $scInfoArray = ConfManager::getScInfo( $sc_id ) ;
        $ret_array = CommonUtility::stepRunProgramAtMs( array(
            "stepname"  => "GetPowerState",
            "program"   => "/usr/bin/vmware-cmd",
            "arguments" => array(
                "--server",   $scInfoArray[ "esxi_ip" ],
                "--username", $scInfoArray[ "esxi_username" ],
                "--password", $scInfoArray[ "esxi_password" ],
                $scInfoArray[ "image_path" ],
                "getstate"
        ) ) ) ;
        
        return join( "\n", $ret_array[ 1 ] ) ;
    }
    
    static function getVmwareToolsState( $sc_id ) {
        $scInfoArray = ConfManager::getScInfo( $sc_id ) ;
        $ret_array = CommonUtility::stepRunProgramAtMs( array(
            "stepname"  => "QueryVmwareToolsStatus",
            "program"   => "/usr/bin/vmware-cmd",
            "arguments" => array(
                "--server",   $scInfoArray[ "esxi_ip" ],
                "--username", $scInfoArray[ "esxi_username" ],
                "--password", $scInfoArray[ "esxi_password" ],
                $scInfoArray[ "image_path" ],
                "gettoolslastactive"
        ) ) ) ;
        
        return join( "\n", $ret_array[ 1 ] ) ;
    }
    
    static function getServiceStatus( $sc_id, $serviceName ) {
        $ret_array = self::stepRunProgramBySsh( $sc_id, array(
            "stepname"  => "QueryServiceStatus: $serviceName, ",
            "program"   => "/sbin/service",
            "arguments" => array(
                $serviceName,
                "status"
        ) ) ) ;
        
        return join( "\n", $ret_array[ 1 ] ) ;
    }
    
    static function getConnectionStatus( $sc_id ) {
        try {
            self::waitUntilSshable( $sc_id, true, array(
                "retry_max"      => 0,
                "retry_interval" => 0 ) ) ;
        }
        catch( Exception $e ) {
            return false ;
        }
        
        return true ;
    }
    
    static function waitUntilPowerState( $sc_id, $criteria, $argv = null ) {
        $retryCount    = 0 ;
        $retryCountMax = isset( $argv[ "retry_max" ] )      ? $argv[ "retry_max" ]      : 30 ;
        $retryInterval = isset( $argv[ "retry_interval" ] ) ? $argv[ "retry_interval" ] : 30 ;
        
        while( true ) {
            $powerState = self::getPowerState( $sc_id ) ;
            CommonUtility::debug_print( "sc_id='$sc_id', powerState='$powerState'" ) ;
            
            if( strcasecmp( $powerState, "getstate() = $criteria" ) === 0 ) {
                break ;
            }
            
            CommonUtility::debug_print( "Sleep $retryInterval seconds..." ) ;
            sleep( $retryInterval ) ;
            
            $retryCount++ ;
            if( $retryCount > $retryCountMax ) {
                throw new Exception( "WaitUntilPowerStateTimeout" ) ;
            }
            
            CommonUtility::debug_print( "retry($retryCount/$retryCountMax)..." ) ;
        }
    }
    
    static function waitUntilVmwareToolsState( $sc_id, $criteria, $argv = null ) {
        $retryCount    = 0 ;
        $retryCountMax = isset( $argv[ "retry_max" ] )      ? $argv[ "retry_max" ]      : 30 ;
        $retryInterval = isset( $argv[ "retry_interval" ] ) ? $argv[ "retry_interval" ] : 30 ;
        
        $statusMeaningArray = array(
            "gettoolslastactive() = 0"   => "VMware Tools is not installed or not running.",
            "gettoolslastactive() = 1"   => "Guest operating system is responding normally.",
            "gettoolslastactive() = 5"   => "Intermittent heartbeat. There might be a problem with the guest operating system.",
            "gettoolslastactive() = 100" => "No heartbeat. Guest operating system might have stopped responding."
        ) ;
        
        while( true ) {
            $vmwareToolsState = self::getVmwareToolsState( $sc_id ) ;
            $statuMeaning = isset( $statusMeaningArray[ $vmwareToolsState ] ) ?
                                   $statusMeaningArray[ $vmwareToolsState ]   :
                                   "unknown" ;
            CommonUtility::debug_print( "sc_id='$sc_id', vmwareToolsState='$vmwareToolsState', meaning='$statuMeaning'" ) ;
            
            if( strcasecmp( $vmwareToolsState, "gettoolslastactive() = $criteria" ) === 0 ) {
                break ;
            }
            
            CommonUtility::debug_print( "Sleep $retryInterval seconds..." ) ;
            sleep( $retryInterval ) ;
            
            $retryCount++ ;
            if( $retryCount > $retryCountMax ) {
                throw new Exception( "WaitUntilVmwareToolsTimeout" ) ;
            }
            
            CommonUtility::debug_print( "retry($retryCount/$retryCountMax)..." ) ;
        }
    }
    
    static function waitUntilSshable( $sc_id, $criteria = true, $argv = null ) {
        $retryCount    = 0 ;
        $retryCountMax = isset( $argv[ "retry_max" ] )      ? $argv[ "retry_max" ]      : 30 ;
        $retryInterval = isset( $argv[ "retry_interval" ] ) ? $argv[ "retry_interval" ] : 30 ;
        
        $scInfoArray = ConfManager::getScInfo( $sc_id ) ;
        
        while( true ) {
            $sshable = CommonUtility::isIpSshable( $scInfoArray[ "ip" ] ) ;
            CommonUtility::debug_print( "sc_id='$sc_id', sc_ip='${scInfoArray[ "ip" ]}', sshable='$sshable'" ) ;
            
            if( $sshable == $criteria ) {
                break ;
            }
            
            CommonUtility::debug_print( "Sleep $retryInterval seconds..." ) ;
            sleep( $retryInterval ) ;
            
            $retryCount++ ;
            if( $retryCount > $retryCountMax ) {
                throw new Exception( "WaitUntilSshableTimeout" ) ;
            }
            
            CommonUtility::debug_print( "retry($retryCount/$retryCountMax)..." ) ;
        }
    }
    
    static function waitUntilServiceState( $sc_id, $serviceName, $criteria, $argv = null ) {
        $retryCount    = 0 ;
        $retryCountMax = isset( $argv[ "retry_max" ] )      ? $argv[ "retry_max" ]      : 30 ;
        $retryInterval = isset( $argv[ "retry_interval" ] ) ? $argv[ "retry_interval" ] : 30 ;
        
        $ptn = "/$criteria/i" ;
        
        while( true ) {
        
            # in some case, getServiceStatus might throw exception due to abnormal service status.
            try {
                $status = self::getServiceStatus( $sc_id, $serviceName ) ;
                CommonUtility::debug_print( "sc_id='$sc_id', status='$status', criteria='$criteria'" ) ;
                
                if( preg_match( $ptn, $status, $matches ) ) {
                    break ;
                }
            }
            catch( Exception $e ) {
                $msg = "[" . basename( $e -> getFile() ) . "(" . $e -> getLine() . ")]: " . $e -> getMessage() . "\n" ;
                CommonUtility::debug_print( $msg ) ;
            }
            
            CommonUtility::debug_print( "Sleep $retryInterval seconds..." ) ;
            sleep( $retryInterval ) ;
            
            $retryCount++ ;
            if( $retryCount > $retryCountMax ) {
                throw new Exception( "WaitUntilServiceStateTimeout" ) ;
            }
            
            CommonUtility::debug_print( "retry($retryCount/$retryCountMax)..." ) ;
        }
    }
    
    static function powerOnById( $sc_id ) {
        $scInfoArray = ConfManager::getScInfo( $sc_id ) ;
        
        # get PowerState
        $powerState = self::getPowerState( $sc_id ) ;
        CommonUtility::debug_print( "sc_id='$sc_id', powerState='$powerState'" ) ;
        
        # power-on when it is off
        if( strcasecmp( $powerState, "getstate() = off" ) == 0 ) {
            CommonUtility::stepRunProgramAtMs( array(
                "stepname"  => "PowerOn",
                "program"   => "/usr/bin/vmware-cmd",
                "arguments" => array(
                    "--server",   $scInfoArray[ "esxi_ip" ],
                    "--username", $scInfoArray[ "esxi_username" ],
                    "--password", $scInfoArray[ "esxi_password" ],
                    $scInfoArray[ "image_path" ],
                    "start"
            ) ) ) ;
        }
        
        # wait PowerState is on
        self::waitUntilPowerState( $sc_id, "on" ) ;
        
        # wait VmwareTools is ready
        self::waitUntilVmwareToolsState( $sc_id, "1" ) ;
    }
    
    static function powerOnByVmpath( $sc_vmpath ) {
        
    }
    
    static function powerOffById( $sc_id ) {
        $scInfoArray = ConfManager::getScInfo( $sc_id ) ;
        
        # get PowerState
        $powerState = self::getPowerState( $sc_id ) ;
        CommonUtility::debug_print( "sc_id='$sc_id', powerState='$powerState'" ) ;
        
        # power-off when it is on
        if( strcasecmp( $powerState, "getstate() = on" ) == 0 ) {
            CommonUtility::stepRunProgramAtMs( array(
                "stepname"  => "PowerOff",
                "program"   => "/usr/bin/vmware-cmd",
                "arguments" => array(
                    "--server",   $scInfoArray[ "esxi_ip" ],
                    "--username", $scInfoArray[ "esxi_username" ],
                    "--password", $scInfoArray[ "esxi_password" ],
                    $scInfoArray[ "image_path" ],
                    "stop",
                    "soft"
            ) ) ) ;
            
            CommonUtility::debug_print( "sleep 10 seconds" ) ;
            sleep( 10 ) ;
        }
        
        # wait PowerState is off
        self::waitUntilPowerState( $sc_id, "off" ) ;
    }
    
    static function powerOffByVmpath( $sc_vmpath ) {
    }

    static function runProgramBySsh( $sc_id, $argv, $confidential_option_list = array()) {
        $scInfoArray = ConfManager::getScInfo( $sc_id ) ;

        $program    = $argv[ 'program' ] ;
        $arguments  = isset( $argv[ 'arguments' ] ) ? $argv[ 'arguments' ] : array() ;
        $isBlocking = isset( $argv[ 'is_block' ] )  ? $argv[ 'is_block' ]  : true ;
        $outputPath = isset( $argv[ 'pipe_to' ] )   ? $argv[ 'pipe_to' ]   : null ;
        $stepname   = isset( $argv[ 'stepname' ] )  ? $argv[ 'stepname' ]  : "runProgramBySsh" ;
        
        # escape arguments
        for( $i = 0 ; $i < count( $arguments ); $i++ ) {
            $arguments[ $i ] = escapeshellarg( $arguments[ $i ] ) ;
        }

        if( $outputPath !== null ) {
            array_push( $arguments, 
                        '>',
                        escapeshellarg( $outputPath ) ) ;
        }        
        
        # compose command
        $command = join( " ", array( $program,
                                     join( " ", $arguments ),
                                     "; echo \$?" ) ) ;
        $mask_command = CommonUtility::mask_confidential_option_list($command, $confidential_option_list);
        CommonUtility::debug_print( "command='$mask_command'" ) ;

        # get ssh connection
        $connection = CommonUtility::getSshConnection(
                            $scInfoArray[ "ip" ],
                            $scInfoArray[ "username" ],
                            $scInfoArray[ "password" ],
                            $scInfoArray[ "port" ]) ;

        # execute
        $stdoutStream = ssh2_exec( $connection, $command ) ;
        $stderrStream = ssh2_fetch_stream( $stdoutStream, SSH2_STREAM_STDERR ) ;

        if( $stdoutStream === false ) {
            $msg = "ExecuteCommandFail: command='$mask_command'" ;
            CommonUtility::debug_print( $msg ) ;
            return array( "1", array( $msg ) ) ;
        }

        # wait for execution result
        $retValue = "0" ;
        $outMsgArray = array() ;
        $errMsgArray = array() ;
        if( $isBlocking === true ) {
            CommonUtility::debug_print( "WaitingForSshResult..." ) ;
            stream_set_blocking( $stdoutStream, 1 ) ;
            
            $outMsgArray = preg_split( "/\n+/", trim( stream_get_contents( $stdoutStream ) ) ) ;
            $errMsgArray = preg_split( "/\n+/", trim( stream_get_contents( $stderrStream ) ) ) ;
            CommonUtility::debug_print( "ReceivedSshResult..." ) ;
            
            $retValue = array_pop( $outMsgArray ) ;
        }
        fclose( $stdoutStream ) ;
        fclose( $stderrStream ) ;

        return array( $retValue,
                      $outMsgArray,
                      $errMsgArray ) ;
    }
    
    static function sendFileBySsh( $argv ) {
        $mach_id  = isset( $argv[ 'mach_id' ] )  ? $argv[ 'mach_id' ]  : "" ;
        $from     = isset( $argv[ 'from' ] )     ? $argv[ 'from' ]     : "" ;
        $to       = isset( $argv[ 'to' ] )       ? $argv[ 'to' ]       : "" ;
        $stepname = isset( $argv[ 'stepname' ] ) ? $argv[ 'stepname' ] : "sendFileBySsh" ;
        
        $scInfoArray = ConfManager::getScInfo( $mach_id ) ;
        
        # get ssh connection
        $connection = CommonUtility::getSshConnection(
            $scInfoArray[ "ip" ],
            $scInfoArray[ "username" ],
            $scInfoArray[ "password" ],
            $scInfoArray[ "port" ]) ;

        # execute
        $retValue = ssh2_scp_send($connection, $from, $to);
        CommonUtility::debug_print( "$stepname, retValue='$retValue'" ) ;
        
        ssh2_exec($connection, 'exit');
        return $retValue;
    }
    
    static function recvFileBySsh( $argv ) {
        $mach_id  = isset( $argv[ 'mach_id' ] )  ? $argv[ 'mach_id' ]  : "" ;
        $from     = isset( $argv[ 'from' ] )     ? $argv[ 'from' ]     : "" ;
        $to       = isset( $argv[ 'to' ] )       ? $argv[ 'to' ]       : "" ;
        $stepname = isset( $argv[ 'stepname' ] ) ? $argv[ 'stepname' ] : "recvFileBySsh" ;
        
        $scInfoArray = ConfManager::getScInfo( $mach_id ) ;
        
        # get ssh connection
        $connection = CommonUtility::getSshConnection(
            $scInfoArray[ "ip" ],
            $scInfoArray[ "username" ],
            $scInfoArray[ "password" ],
            $scInfoArray[ "port" ]) ;

        # execute
        $retValue = ssh2_scp_recv($connection, $from, $to);
        CommonUtility::debug_print( "$stepname, retValue='$retValue'" ) ;
        
        ssh2_exec($connection, 'exit');
        return $retValue;
    }

    static function runProgramByVmware( $sc_id, $argv ) {
        $scInfoArray = ConfManager::getScInfo( $sc_id ) ;
        
        $program    = $argv[ 'program' ] ;
        $arguments  = isset( $argv[ 'arguments' ] ) ? $argv[ 'arguments' ] : array() ;
        $outputPath = isset( $argv[ 'pipe_to' ] )   ? $argv[ 'pipe_to' ]   : null ;
        $isEscape   = isset( $argv[ 'escape' ] )    ? $argv[ 'escape' ]    : 1 ;
        $stepname   = isset( $argv[ 'stepname' ] )  ? $argv[ 'stepname' ]  : "runProgramByVmware" ;
        
        $newArgumentsArray = array(
            "-T",  "server",
            "-h",  $scInfoArray[ "esxi_ip" ],
            "-u",  $scInfoArray[ "esxi_username" ],
            "-p",  $scInfoArray[ "esxi_password" ],
            "-gu", $scInfoArray[ "username" ],
            "-gp", $scInfoArray[ "password" ],
            "runProgramInGuest",
            $scInfoArray[ "image_path_vix" ],
            $program
        ) ;
        
        foreach( $arguments as $argument ) {
            array_push( $newArgumentsArray, $argument ) ;
        }
        
        CommonUtility::stepRunProgramAtMs( array(
            "stepname"  => $stepname,
            "pipe_to"   => $outputPath,
            "escape"    => $isEscape,
            
            "program"   => "/usr/bin/vmrun",
            "arguments" => $newArgumentsArray
        ),
            array('-u', '-p', '-gu', '-gp')
        ) ;
    }
    
    static function runProgram( $sc_id, $argv ) {
        $scInfoArray = ConfManager::getScInfo( $sc_id ) ;
        
        if( CommonUtility::isIpSshable( $scInfoArray[ "ip" ] ) == true ) {
            return self::runProgramBySsh( $sc_id, $argv ) ;
        }
        else {
            return self::runProgramByVmware( $sc_id, $argv ) ;
        }
    }
    
    static function stepRunProgramBySsh( $sc_id, $argv, $confidential_option_list = array() ) {
        $stepName = isset( $argv[ "stepname" ] ) ? $argv[ "stepname" ] : "RunProgramAtSc" ;

        # introducing retry mechanism
        $retryCount    = 0 ;
        $retryCountMax = isset( $argv[ "retry_max" ] )      ? $argv[ "retry_max" ]      : 3 ;
        $retryInterval = isset( $argv[ "retry_interval" ] ) ? $argv[ "retry_interval" ] : 30 ;
        
        # composing debug message
        $msg = $stepName . ": " . escapeshellarg( $argv[ 'program' ] ) ;
        $argumentsArray = isset( $argv[ 'arguments' ] ) ? $argv[ 'arguments' ] : array() ;
        foreach( $argumentsArray as $argument ) {
            $msg .=  " " . escapeshellarg( $argument ) ;
        }
        $msg = CommonUtility::mask_confidential_option_list($msg, $confidential_option_list);
        CommonUtility::debug_print( $msg ) ;
        
        # check if run at all SC,
        #    0: means all SC
        $sc_id_list = array( $sc_id ) ;
        if( $sc_id == "0" ) {
            $sc_id_list = ScManager::getScIdList() ;
        }
        dprint( 0, LOG_DEBUG, "sc_id_list='" . join( ",", $sc_id_list ) . "'" ) ;
        
        # launching
        foreach( $sc_id_list as $sc_id ) {
            $exceptionMsg = "" ;
            while( true ) {
                try {
                    $ret_array = self::runProgramBySsh( $sc_id, $argv, $confidential_option_list ) ;
                    
                    # verify return code
                    $retCode      = $ret_array[ 0 ] ;
                    $stdoutMsgAry = $ret_array[ 1 ] ;
                    $stdoutMsg    = join( "\n", $stdoutMsgAry ) ;
                    
                    if( $retCode == 0 ) {
                        CommonUtility::debug_print( "${stepName}Pass: msg='$stdoutMsg'" ) ;
                        return $ret_array ;
                    }
                    
                    $stderrMsgAry = $ret_array[ 2 ] ;
                    $stderrMsg    = join( "\n", $stderrMsgAry ) ;
                    CommonUtility::debug_print( "${stepName}Fail: ret='$retCode', msg='$stderrMsg'" ) ;
                }
                
                catch( Exception $e ) {
                    $exceptionMsg = "[" . basename( $e -> getFile() ) . "(" . $e -> getLine() . ")]: " . $e -> getMessage() . "\n" ;
                    CommonUtility::debug_print( $exceptionMsg ) ;
                }
                
                CommonUtility::debug_print( "Sleep $retryInterval seconds..." ) ;
                sleep( $retryInterval ) ;
                
                $retryCount++ ;
                if( $retryCount > $retryCountMax ) {
                    throw new Exception( $exceptionMsg ) ;
                }
                
                CommonUtility::debug_print( "retry($retryCount/$retryCountMax)..." ) ;
            }
        }
    }
    
    static function stepSendFile( $mach_id, $argv, $confidential_option_list = array() ) {
        $stepName = isset( $argv[ "stepname" ] ) ? $argv[ "stepname" ] : "SendFileToSc" ;

        # check if run at all SC,
        #    0: means all SC
        $mach_id_list = array( $mach_id ) ;
        if( $mach_id == "0" ) {
            $mach_id_list = ScManager::getScIdList() ;
        }
        dprint( 0, LOG_DEBUG, "mach_id_list='" . join( ",", $mach_id_list ) . "'" ) ;
        
        # launching
        foreach( $mach_id_list as $mach_id ) {
            $argv['mach_id'] = $mach_id;
        
            CommonUtility::debug_print( $stepName ) ;
            CommonUtility::retry( 'ScManager::sendFileBySsh', $argv );
        }
    }
    
    static function stepRecvFile( $mach_id, $argv, $confidential_option_list = array() ) {
        $stepName = isset( $argv[ "stepname" ] ) ? $argv[ "stepname" ] : "ReceiveFileFromSc" ;
        $argv['mach_id'] = $mach_id;
        
        CommonUtility::debug_print( $stepName ) ;
        CommonUtility::retry( 'ScManager::recvFileBySsh', $argv );
    }
    
    static function setNetworkConfiguration( $sc_id, $argv ) {
        $scInfoArray = ConfManager::getScInfo( $sc_id ) ;
        
        $nic     = isset( $argv[ "nic" ] )     ? $argv[ "nic" ]     : "eth0" ;
        $dhcp    = isset( $argv[ "dhcp" ] )    ? $argv[ "dhcp" ]    : "1" ;
        $ip      = isset( $argv[ "ip" ] )      ? $argv[ "ip" ]      : null ;
        $mask    = isset( $argv[ "mask" ] )    ? $argv[ "mask" ]    : null ;
        $gateway = isset( $argv[ "gateway" ] ) ? $argv[ "gateway" ] : null ;
        $dns     = isset( $argv[ "dns" ] )     ? $argv[ "dns" ]     : null ;
        
        $ipConf = "" ;
        if( $dhcp == "1" ) {
            $ipConf = "DEVICE=$nic\n" .
                      "BOOTPROTO=dhcp\n" . 
                      "ONBOOT=yes\n" ;
        }
        else {
            $ipConf = "DEVICE=$nic\n" .
                      "BOOTPROTO=static\n" . 
                      "ONBOOT=yes\n" .
                      "IPADDR=$ip\n" .
                      "NETMASK=$mask\n" .
                      "GATEWAY=$gateway\n" ;
        }

        self::runProgramByVmware( $sc_id, array(
            "stepname"  => "ModifyScNetworkConf: nic='$nic', ",
            "pipe_to"   => "/etc/sysconfig/network-scripts/ifcfg-$nic",
            "program"   => "/bin/echo",
            "arguments" => array( $ipConf )
        ) ) ;
        
        self::runProgramByVmware( $sc_id, array(
            "stepname"  => "StopScNic: nic='$nic', ",
            "program"   => "/sbin/ifdown",
            "arguments" => array( $nic )
        ) ) ;
        
        self::runProgramByVmware( $sc_id, array(
            "stepname"  => "StartScNic: nic='$nic', ",
            "program"   => "/sbin/ifup",
            "arguments" => array( $nic )
        ) ) ;
        
        # self::waitUntilSshable( $sc_id ) ;
        
        self::runProgramByVmware( $sc_id, array(
            "stepname"  => "UpdateDNS",
            "pipe_to"   => "/etc/resolv.conf",
            "program"   => "/bin/echo",
            "arguments" => array( "nameserver $dns\n" )
        ) ) ;
    }

    static function controlService( $sc_id, $serviceName, $action ) {
        
    }
}

if( realpath( $_SERVER[ "SCRIPT_NAME" ] ) == realpath( __FILE__ ) ) {
    require_once( "Constants.class.php" ) ;
    require_once( "CommonUtility.class.php" ) ;
    require_once( "ConfManager.class.php" ) ;
    require_once( "DbManager.class.php" ) ;

    try {
        // ScManager::powerOffById( "1" ) ;
        // ScManager::powerOnById( "1" ) ;
        // ScManager::runProgramByVmware( "1", array(
            // "program" => "/bin/touch",
            // "arguments" => array(
                // "/tmp/test1",
                // "/tmp/test2"
        // ) ) ) ;
        // ScManager::setNetworkConfiguration( "1", array(
            // "nic"     => "eth0",
            // "dhcp"    => "0",
            // "ip"      => "169.254.3.3",
            // "mask"    => "255.255.255.0",
            // "gateway" => "169.254.3.2",
            // "dns"     => "169.254.3.2"
        // ) ) ;
        
        var_dump( ScManager::getScIdList() ) ;
    }
    
    catch( Exception $e ) {
        echo "[" . $e -> getFile() . "(" . $e -> getLine() . ")]: " . $e -> getMessage() . "\n" ;
    }
}
